#include "gpslib.h"
#include <dstring.h>
#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <qpixmap.h>


class maplist: public list_of<mapinfo>{
public:
	maplist():mrp(0){}
	~maplist();
	const char *readmapdata(const char *mapname);
	// Messy iterator-like function. Returns pointer to
	// maps which match
	mapinfo * locatemap(double lat, double longit);
	const mapinfo * locatemap(const char *cp);
	// If a map was located, here's the gen
	const mapinfo *mp()	{ return mrp; }
   void clear(){list_of<mapinfo>::clear(); mrp = 0;}
private:
	// last item found by locatemap (if any)
	mapinfo *mrp;
} mapinfolist;

int initmaplist(const char *mapname){
	mapinfolist.clear();
	const char *cp = mapinfolist.readmapdata(mapname);
	if(cp){
		printwhinge("Error Reading Map Data: %s", cp);
      return 0;
	}
	return 1;
}

double requestedscale = 0;
void locatescale(double pixperdegn){
	requestedscale = pixperdegn;
}

int enumeratemaps(double lat, double longit, char *namep, int *namelen, double *nscalep, double *escalep){
	const mapinfo *mp = mapinfolist.locatemap(lat, longit);
	if(mp){
		if(namep) strcpy(namep, mp->name.cstring());
		if(namelen) *namelen = mp->name.nchars();
		if(nscalep) *nscalep = mp->pixperdegheight;
		if(escalep) *escalep = mp->pixperdegwidth;
		return 1;
	}
	return 0;
}

// locatemap scans for candidate maps, returning a "best" fit. Here
// we remember the scales found so that other options can be shown
sortable_list_of<double>nscalesfound;

// Call enumeratescales if locatemap has returned true. Returns
// available scales (pix per deg N)
void enumeratescales(list_of<double> &lodr){
	lodr.clear();
	const ne = nscalesfound.element_count();

	// return in reverse order
	for(int i = ne-1; i >= 0; i--){
		lodr += nscalesfound[i];
	}
}

int locatemap(double lat, double longit, String &name, double &nscale, double &escale){
	if(requestedscale == 0){
		const mapinfo *mp = mapinfolist.locatemap(lat, longit);
		if(mp){
			name = mp->name;
			nscale = mp->pixperdegheight;
			escale = mp->pixperdegwidth;
			return 1;
		}
		return 0;
	}
	const mapinfo *mp = locatemapinfo(lat, longit);
	if(mp == 0)
		return 0;
	name = mp->name;
	nscale = mp->pixperdegheight;
	escale = mp->pixperdegwidth;
	return 1;
}

const mapinfo * locatemapinfo(double lat, double longit){
	if(requestedscale == 0)
		return mapinfolist.locatemap(lat, longit);

	// Requestedscale isn't zero - first find 'best' scale, then of all maps
	// within 5%, return the one with the nearest centre point
	nscalesfound.clear();
	// char buff[512];
	mapinfo *mip;
	if(verbose)printf("locatemapinfo, requested scale is %f\n", requestedscale);

	double bestdiff = -100;
	// For each map that contains the desired point
	while((mip = mapinfolist.locatemap(lat, longit)) != 0){
		double thisdiff = fabs(requestedscale-mip->pixperdegheight);
		if(bestdiff < 0 || thisdiff < bestdiff)
			bestdiff = thisdiff;

		// gotalready tells us if we have this approximately this scale in the list
		int gotalready = 0;
		for(int i = 0; i < nscalesfound.element_count(); i++){
			if(fabs(nscalesfound[i]-mip->pixperdegheight) < mip->pixperdegheight/20){
				mip->scalefamily = nscalesfound[i];
				gotalready++;
				break;
			}
		}
		if(!gotalready){
			nscalesfound += mip->pixperdegheight;
			mip->scalefamily = mip->pixperdegheight;
			if(verbose)printf("Adding %f to scale list\n", mip->pixperdegheight);
		}
	}
	nscalesfound.sort();
	if(verbose)printf("Located maps with %d different scales\n", nscalesfound.element_count());

	const mapinfo *mp = 0;
	double bestmiddiff = -100;
	while((mip = mapinfolist.locatemap(lat, longit)) != 0){
		double thisdiff = fabs(requestedscale-mip->pixperdegheight);
		// printf("map scale %f bestdiff %f, thisdiff %f\n", mip->pixperdegheight, bestdiff, thisdiff);
		if(fabs(thisdiff-bestdiff) <= bestdiff/20){
			// prepared to use this one
			// printf("%f within range, family %f\n", thisdiff-bestdiff, mip->scalefamily);
			double latdiff = lat-(mip->latmin+(mip->latmax-mip->latmin)/2);
			double longdiff = longit-(mip->longmin+(mip->longmax-mip->longmin)/2);
			double distance = sqrt(latdiff*latdiff+(longdiff*longdiff));
			if(bestmiddiff < 0 || distance < bestmiddiff){
				// mip->requestedscale = requestedscale;
				mp = mip;
				bestmiddiff = distance;
				// strcpy(buff, mip->name.cstring());
			}
		}
	}
	return mp;

}

#ifdef BITMAP
class bitmap{
public:
	enum howget{full, justinfo};
	bitmap(const char *fname, howget get=justinfo);
	bitmap();
	~bitmap();
	int reload(const string &fname);
	int isvalid()const			{ return vf; }
	long height()const		{ return vf ? dibpixheight : 0;}
	long width()const		{ return vf ? dibpixwidth : 0;}
	// drawon a window
	int drawon(int hwnd, int bmx, int bmy, int windx, int windy, int width, int height);
private:
	int vf;					// validflag
	char huge *bmdp;		// the data pointer whilst reading from file
	long dibpixwidth, dibpixheight;	// pixels w/h
	unsigned long bits;		// offset into bmdp where the bits live
	int hbitmap;
	// we can't copy these things around, sorry
	bitmap(const bitmap&);
	void operator =(const bitmap&);
	void doload(const char *fname, howget);
};

inline int bitmap::reload(const string &fname){
	doload(fname.cstring(), full);
	return vf;
}

int showmap(const char *name, double midlat, double midlong, int hwnd){
	const mapinfo *mip = mapinfolist.locatemap(name);
	if(mip == 0){
		MessageBox(0, "Requested map name not found in showmap", name,  MB_ICONSTOP);
		return 0;
	}
	// Bitmap cache
	static string mapfilename;
	static bitmap bm;
	if(mapfilename != name){
   	mapfilename = name;
		bm.reload(mapfilename);
	}

	if(!bm.isvalid()){
		MessageBox(0, "Requested bitmap invalid in showmap", name,  MB_ICONSTOP);
		return 0;
	}
	if(midlat < mip->latmin || midlat > mip->latmax || midlong < mip->longmin || midlong > mip->longmax){
		MessageBox(0, "midlat/long out of range!", "showmap", MB_ICONSTOP);
      return 0;
	}
	RECT r;
	GetClientRect(hwnd, &r);
	// Locate the pixel within the bitmap which corresponds to midlat/long
	// This was originally done by extrapolating lat/long figures, but that
	// doesn't work too well because of projection errors (NGR grid does not
	// follow lat/long very well).
	int midlatpix, midlongpix;
	if(mip->osproj){
		double east=0, north=0;
		cvtongr(midlat, midlong, 0, &east, &north);
		midlatpix = (mip->northmax-north)*mip->pixpermetreheight;
		midlongpix = (east-mip->eastmin)*mip->pixpermetrewidth;
	}else{
		midlatpix = (mip->latmax-midlat)*mip->pixperdegheight;
		midlongpix = (midlong-mip->longmin)*mip->pixperdegwidth;
	}
	// Adjust offsets within pixel regions
	int wndstartx, wndstarty, mapstartx, mapstarty;
	// work out where on the window the left and top map edges lie - adjust
	// if outside
	wndstartx = r.right/2-midlongpix;
	wndstarty = r.bottom/2-midlatpix;
	mapstartx = 0; mapstarty = 0;
	if(wndstartx < 0){
		mapstartx -= wndstartx;
		wndstartx = 0;
	}
	if(wndstarty < 0){
		mapstarty -= wndstarty;
		wndstarty = 0;
	}
	// and now the drawing width and height
	int dwidth, dheight;
	dwidth = r.right - wndstartx;
	dheight = r.bottom - wndstarty;
	if(dwidth > mip->pixwidth-mapstartx)
		dwidth = int(mip->pixwidth-mapstartx);
	if(dheight > mip->pixheight-mapstarty)
		dheight = int(mip->pixheight-mapstarty);
#if REPORTMAP
	char msgbuff[512];
	sprintf(msgbuff, "mapstartx %d, mapstarty %d, wndstartx %d, wndstarty %d, width %d, height %d",
				mapstartx, mapstarty, wndstartx, wndstarty, dwidth, dheight);
	MessageBox(0, msgbuff, "showmap", MB_ICONSTOP);
#endif
	return bm.drawon(hwnd, mapstartx, mapstarty, wndstartx, wndstarty, dwidth, dheight);
}

#endif

void mapinfo::print()const{
	printf("mapinfo: map %s, longmin %f longmax %g latmin %g latmax %g\n",
		name.cstring(), longmin, longmax, latmin, latmax);
	printf("eastmin %g, eastmax %g, northmin %g, northmax %g\n", eastmin, eastmax, northmin, northmax);
	printf("pixwidth %ld, pixheight %ld, pixperdegwidth %g, pixperdegheight %g\n", pixwidth, pixheight, pixperdegwidth, pixperdegheight);
	printf("pixpermetrewidth %g, pixpermetreheight %g\n",pixpermetrewidth, pixpermetreheight);
}

mapinfo * maplist::locatemap(double lat, double longit){
	static counter;
	while(counter < element_count()){
		mapinfo &mr = operator[](counter);
		counter++;
		if(lat < mr.latmax && lat > mr.latmin && longit < mr.longmax && longit > mr.longmin){
			mrp = &mr;
			return mrp;
		}
	}
	counter = 0;
	mrp = 0;
	return 0;
}

const mapinfo * maplist::locatemap(const char *cp){
	for (int counter = 0; counter < element_count(); counter++){
		mapinfo &mr = operator[](counter);
		if(strcmp(cp, mr.name.cstring()) == 0){
			return &mr;
		}
	}
	return 0;
}

maplist::~maplist(){
	//printf("Destroy a maplist\n");
}

const char * maplist::readmapdata(const char *mapname){
	FILE *fp = fopen(mapname, "r");
	String str;
	static char ebuff[512];

	if(!fp){
		sprintf(ebuff, "Can't open %s", mapname);
		return ebuff;
	}

	while(fgets(ebuff, sizeof ebuff, fp)){
		// printf("Read string %s\n", str.cstring());
		if(ebuff[0] == '#')
			continue;
		String str = ebuff;
		array_of_String strar, namesize;
		strar.getfields(str, ",");
		const ne = strar.element_count();
		if(ne == 0)
			continue;
		if(ne != 7 && ne != 9){
			sprintf(ebuff, "Bad field count %d on line \"%s\"", ne, str.cstring());
			fclose(fp);
			return ebuff;
		}

		// Get the gen - first field contains colon-separated sizes if any
		namesize.getfields(strar[0], ":");
		mapinfo minf;
		minf.name = namesize[0];
		minf.osproj = (ne == 7);
		if(namesize.element_count() == 1){
			QPixmap bm;
			if(bm.load(minf.name.cstring())){
				minf.pixwidth = bm.width();
				minf.pixheight = bm.height();
				if(verbose) printf("Map name %s has width %d height %d\n", minf.name.cstring(), bm.width(), bm.height());
			}else{
				sprintf(ebuff, "File %s: invalid bitmap format\n",strar[0].cstring());
				fclose(fp);
				return ebuff;
			}
		}else{
			minf.pixwidth = int(namesize[1]);
			minf.pixheight = int(namesize[2]);
				if(verbose) printf("Map name %s has preset width %d height %d\n", minf.name.cstring(), minf.pixwidth, minf.pixheight);

		}
		unsigned long x1, y1, x2, y2;
		double lat1, long1, lat2, long2;
		long east1, north1, east2, north2;
		for(int i = 1; i < ne; i++){
			const String &csr = strar[i];
			const char *cp = csr.cstring();
			// printf("Field %d, %s\n", i, cp);
			while(isspace(*cp))
				cp++;
			// Now process each field
			switch(i){
				case 1:
					x1 = strtol(cp, 0, 10);
					break;
				case 2:
					y1 = strtol(cp, 0, 10);
					break;
				case 3:
					if(ne == 7){
						cvtngrstr(cp, &east1, &north1, &lat1, &long1);
					}else{
						lat1 = getlatitude(cp);
					}
					break;
				case 4:
					if(ne == 7){
						x2 = strtol(cp, 0, 10);
					}else{
						long1 = getlongitude(cp);
						double e, n;
						String dummy;
						cvtongr(lat1, long1, dummy, e, n);
						east1 = (long)e; north1 = (long)n;
					}
					break;
				case 5:
						(ne == 7 ? y2 : x2) = strtol(cp, 0, 10);
					break;
				case 6:
					if(ne == 7){
						cvtngrstr(cp, &east2, &north2, &lat2, &long2);
					}else{
						y2 = strtol(cp, 0, 10);
					}
					break;
				case 7:
					lat2 = getlatitude(cp);
					break;
				case 8:{
					long2 = getlongitude(cp);
					double e, n;
					String dummy;
					cvtongr(lat2, long2, dummy, e, n);
					east2 = (long)e; north2 = (long)n;
					break;
					}
			}
		}
		// printf("Result is\te1 %ld, n1 %ld, %f:%f\n\te2 %ld, n2 %ld, %f:%f\n", east1, north1, lat1, long1, east2, north2, lat2, long2);
		long xdiff = labs(long(x2) - long(x1));
		double longitdiff = fabs(long2-long1);
		long eastdiff = labs(east2-east1);
		// printf("xdiff %ld, longitdiff %g, eastdiff %ld\n", xdiff, longitdiff, eastdiff);
		long ydiff = labs(long(y2) - long(y1));
		double latdiff = fabs(lat2-lat1);
		long northdiff = labs(north2-north1);
		// printf("ydiff %ld, latdiff %g, northdiff %ld\n", ydiff, latdiff, northdiff);
		double xscale = longitdiff/xdiff, yscale = latdiff/ydiff;
		double escale = double(eastdiff)/xdiff, nscale = double(northdiff)/ydiff;

		minf.pixperdegwidth = 1/xscale;
		minf.pixperdegheight = 1/yscale;
		minf.latmax = lat1+y1*yscale;
		minf.latmin = lat2-(minf.pixheight-y2)*yscale;
		minf.longmax = long2+(minf.pixwidth-x2)*xscale;
		minf.longmin = long1-x1*xscale;
		minf.eastmin = east1-x1*escale;
		minf.eastmax = east2+(minf.pixwidth-x2)*escale;
		minf.northmax = north1+y1*nscale;
		minf.northmin = north2-(minf.pixheight-y2)*nscale;
		minf.pixpermetrewidth = 1/escale;
		minf.pixpermetreheight = 1/nscale;
		// minf.print();
		*this += minf;

		//const double deg2rad = atan2(1,1) / 45;
		//double latrad = (lat2 + (lat1-lat2)/2)*deg2rad;
		//double coslat = cos(latrad);

		// printf("deg/pixel: Xscale %g, Yscale %g\n", xscale, yscale);
		// printf("metres/pixel: escale %g , nscale %g\n", escale, nscale);

		//printf("coslat %g, Expect Xscale to be approx %g\n", coslat, yscale/coslat);
		//printf("Scale error is %g%%\n", fabs(xscale-yscale/coslat)*100/xscale);

	}
   fclose(fp);
	return 0;
}

#ifdef BITMAP
bitmap::~bitmap(){
	if(bmdp){GlobalFreePtr(bmdp); bmdp=0;}
	if(hbitmap)DeleteObject(hbitmap);
}
bitmap::bitmap() : vf(0), bmdp(0), hbitmap(0), bits(0){
}
bitmap::bitmap(const char *fname, howget get) : vf(0), bmdp(0), hbitmap(0), bits(0){
	doload(fname, get);
}

void bitmap::doload(const char *fname, howget get){
	if(bmdp){GlobalFreePtr(bmdp); bmdp=0;}
	if(hbitmap){DeleteObject(hbitmap); hbitmap=0;}

	vf = 1;
	bits = 0;
// Read a .bmp file
	int fdes = _lopen(fname, READ);
	// printf("File open returns %d\n", fdes);
	if(fdes < 0){
		vf = 0;
		return;
	}

	char buff[sizeof(BITMAPFILEHEADER)+sizeof(BITMAPINFOHEADER)];
	//BITMAPFILEHEADER bmfh;
	//BITMAPINFOHEADER bmih;
	unsigned rres = _lread(fdes, buff, sizeof buff);
	_lclose(fdes);
	fdes = -1;

	// printf("Read returns %u\n", rres);
	if(rres == sizeof buff){
		BITMAPFILEHEADER *bmfhp = (BITMAPFILEHEADER *)buff;
		// printf("type %.2s, size %lu, offset %lu\n", &(bmfhp->bfType), bmfhp->bfSize, bmfhp->bfOffBits);

		BITMAPINFOHEADER *bihp = (BITMAPINFOHEADER *)(buff + sizeof *bmfhp);
		unsigned long bihsize;
		bihsize = bihp->biSize;
		if(bihsize == (unsigned long)sizeof (BITMAPINFOHEADER)){
			// It's the right size, so pick up the other bits
			dibpixwidth = bihp->biWidth;
			dibpixheight = bihp->biHeight;
			int bitcount = bihp->biBitCount;
			unsigned long ncolours = bihp->biClrUsed;
			if(ncolours == 0 && bitcount != 24)
				ncolours = 1L << bitcount;
			// printf("w %lu, h %lu, bits %u, nc %lu\n", dibpixwidth, dibpixheight, bitcount, ncolours);
			unsigned long coltabsize = ncolours*sizeof(RGBQUAD);
			// printf("colour table size %lu\n", coltabsize);
			bits = coltabsize+bihsize;
			assert(bits == bmfhp->bfOffBits - sizeof *bmfhp);
		}else{
			vf = 0;
		}
	}else{
		vf = 0;
	}
	if(get == justinfo || vf == 0) return;

	/////////////////////////////////////////////////////////////
	// Get the full monty
	fdes = _lopen(fname, READ);
	// printf("File open returns %d\n", fdes);
	if(fdes < 0){
		vf = 0;
		return;
	}
	BITMAPFILEHEADER *bmfhp = (BITMAPFILEHEADER *)buff;
	_lread(fdes, buff, sizeof(*bmfhp)); // skip the unwanted bit

	unsigned long bmsz = bmfhp->bfSize-sizeof(*bmfhp);
	bmdp = GlobalAllocPtr(GMEM_MOVEABLE, bmsz);
	if(bmdp == 0 || _hread(fdes, bmdp, bmsz) != bmsz){
		vf = 0;
	}
	_lclose(fdes);
	if(vf == 0 && bmdp){GlobalFreePtr(bmdp); bmdp=0;}
}

int bitmap::drawon(int hwnd, int bmx, int bmy, int windx, int windy, int width, int height){
	HDC hdc = GetDC(hwnd);
	int rval = 0;
	if(hdc){
		RECT r;
		GetClientRect(hwnd, &r);
		if(hbitmap == 0){
			if(bmdp == 0)
				return 0;

			hbitmap = CreateDIBitmap(	hdc,
												(LPBITMAPINFOHEADER)bmdp,
												CBM_INIT,
												bmdp+bits,
												(LPBITMAPINFO)bmdp,
												DIB_RGB_COLORS);
		}
		if(hbitmap){
				GlobalFreePtr(bmdp); bmdp=0;
				lDrawBitMapsz(hdc, hbitmap,
					windx, windy,	// dest x, y
					bmx, bmy,	// src x, y
					width, height,	// width, height
					SRCCOPY);
					rval = 1;
		}else{
			rval = 0;
		}
		ReleaseDC(hwnd,hdc);
	}
	return rval;
}
#endif
