#ifndef GPS_H
#define GPS_H
#include <qapplication.h>
#include <qlabel.h>
#include <qpixmap.h>
#include <qpainter.h>
#include <qradiobutton.h>
#include <stdio.h>
#include <qmenubar.h>
#include <qpopupmenu.h>
#include <qmessagebox.h>
#include <qtimer.h>
#include <qbuttongroup.h>
#include <dstring.h>
#include "gpslib.h"

inline int max(int a, int b){ return a>b?a:b; }

// The global pointer to GPS
class GPS;
extern GPS *Gps;


struct Position{
	Position(double lat=0, double lng=0) : latit(lat), longit(lng){}
	double latit, longit;
};

struct highlight{
	Position pos;
	String description, type;
};

extern list_of<highlight> highlightlist;
extern highlight home;

// Mapwindow - the class that manages the display of the map
class Mapwindow: public QFrame{
public:
	Mapwindow(GPS *gpsp);

	// Set an initial lat, long
	void home(const highlight &);

	// Move the map to centre on this point
	// Return true if there was a map, false else
	bool positionto(const Position &cpr);

	// Add a point to the track. May not reposition the map
	// unless necessary (track moved far enough from map centre)
	// The offcentrerange parameter says how far from the centre
	// the track may be before a map reposition
	void trackto(double latit, double longit, double utctime, int offcentrerange);
	void cleartrack();

	void setpreferredmapscale(double);
protected:
	void paintEvent(QPaintEvent *);
	void mouseMoveEvent(QMouseEvent *);
	void mousePressEvent(QMouseEvent *);
private:
	GPS *myGpsp;

	// Set the map file name
	void load(const char *cp);

	// this string non-null if we have a map showing
	String currentmapname;
	bool llvalid;	// Lat/Long valid

	// Where we keep our recent track if any. Max size in enum maxtrack.
	list_of<Position> track;
	enum{maxtrack = 1000};
	QPoint lasttrackpoint;	// Last track point drawn *relative to current map*

	Position mappos;	// Lat long of map midpoint showing
	// if there is a valid map, points to it
	const mapinfo *mymapinfo;
	mapinfo mydummymapinfo;	// Used when we can't find a real map

	// Figure out where on the map (pixels) a give lat/long should appear
	// NB - this is position on the current map, not the display surface
	QPoint pos_to_mappixels(const Position &cpr);
	// And now turn a mappixel into a displaypixel
	QPoint mappixel_to_displaypixels(const QPoint &cpr);

	QPixmap mymap;
	// This is set if we need an offset. It holds the
	// pixel coordinates of what should be the middle of
	// the map
	QPoint picoffset;
	// draw the map at the needed offsets for middle to be
	// correct
	void drawoffsetmap();

	void updateGPSpos(const Position &);

	// Return a heading in degrees, given x/y lengths travelled
	double yxheading(double northdiff, double eastdiff);

	// Given Lat/Long & time differences, maintain a smoothed velocity (speed/heading)
	// Uses lastplotpos and lastplottime to figure out rates of change.
	// Also uses n-pole filter to smooth the calculated rate of change
	// THIS MUST ONLY BE CALLED FROM trackto. It relies on cleartrack properly
	// resetting the filters etc - bit of a bodge.
	int plotspeed(const Position &cpr, double utctime);
	bool firstplot;
	const npoles = 3;
	double xvfilter[npoles], yvfilter[npoles];
	const double fconst=0.3;	// rate of roll-off, lower for higher attenuation
	Position lastplotpos;
	double lastplottime;

	double usemapscale;	// Preferred map scale; acted on positionto() only
};


// GPS - the class that coordinates the whole show
class GPS: public QWidget{
	Q_OBJECT
public:
	GPS();
	bool event(QEvent *ep);
protected:
	// void mouseMoveEvent(QMouseEvent *);
	void resizeEvent(QResizeEvent *);
private slots:
	void slotAbout();
	void slotReplay();
	void slotPlot();
	void slotReplaytick();	// The timer ticked
	void slotPlottick();	// The timer ticked
	void slotMaptoggle(int);	// Map scale changed
private:

	QTimer *replaytimer;	// If replaying a log

	// Fixed positions of widgets
	enum{
		leftbarwidth = 80,
		lhwidgets_x = 0,
		lhwidgetsystart = 10,
		lhlabels_h = 20,
		lhlabelsspacing = 5,

		startheight=400,
		startwidth = 400+leftbarwidth,

		mymap_x=leftbarwidth+1,
		mymap_y=0,
		mymap_w = startwidth-mymap_x,
		mymap_h = startheight,

		// Mouse tracking
		mymousex_x=lhwidgets_x,
		mymousex_y=lhwidgetsystart,
		mymousex_w=leftbarwidth,
		mymousex_h=lhlabels_h,

		mymousey_x=lhwidgets_x,
		mymousey_y=mymousex_y+mymousex_h+lhlabelsspacing,
		mymousey_w=leftbarwidth,
		mymousey_h=lhlabels_h,

		altpos_x=lhwidgets_x,
		altpos_y=mymousey_y+lhlabels_h+lhlabelsspacing,
		altpos_w=leftbarwidth,
		altpos_h=lhlabels_h,

		utc_x=lhwidgets_x,
		utc_y=altpos_y+lhlabels_h+lhlabelsspacing,
		utc_w=leftbarwidth,
		utc_h=lhlabels_h,

		speed_x=lhwidgets_x,
		speed_y=utc_y+lhlabels_h+lhlabelsspacing,
		speed_w=leftbarwidth,
		speed_h=lhlabels_h,

		hdng_x=lhwidgets_x,
		hdng_y=speed_y+lhlabels_h+lhlabelsspacing,
		hdng_w=leftbarwidth,
		hdng_h=lhlabels_h,

		scales_y = hdng_y+lhlabels_h+lhlabelsspacing,
		scales_x = lhwidgets_x,
	};

	// mysubwidgets
	QLabel *mymousex, *mymousey, *altpos, *utc, *speed, *hdng;
	QMenuBar * menubar;
	int mbht;	// menubar height
	QPopupMenu * filemenu;
	int replayid;	// Id of file/replay
	int plotid;	// Id of file/plot (i.e. listen to receiver)
	QPopupMenu * helpmenu;

	list_of<double>mapscales;	// currently available scales
	QLabel *scale;
	list_of_ptr_to<QRadioButton>scalebuttonptrs;
	void showmapscales(const mapinfo *);		// once mapscales set, display radiobuttons
	QButtonGroup *scalegrp;

	Mapwindow *mymap;
	friend class Mapwindow;

	bool plotting;
	void stopplotreplay();	// reset timer etc.
	void startplotreplay(); // start timer etc. Uses plotting variable
	int asyncfdes;		// descriptor of async line used when reading plots

};
#endif
