#ifndef STRING_H
#define STRING_H
#include <stdlib.h>
#include <string.h>
#include <genarray.h>

#pragma intrinsic strlen

class String{
	struct String_data{	// reference-counted data field
		int refcnt;
		char data[1];	// actually, usually more than one
		const char *cstring()			{return data;}
		void operator++();
		void operator--();
	};

	static String_data *newsdp(size_t);
	static String_data *newsdp(String_data *);

	union{
		String_data *sdp;
		String *freeptr;
	};
	void init(const char *ccp);
	void single_copy();	// ensure that we own a single copy of our data
#ifdef STRING_POOL_ALLOC
	static String *freelist;	// pool allocation
public:
	void * operator new(size_t);
	void operator delete(void *, size_t);
#endif

	// Use great care with this! It allows construction of a String from
	// an already-allocated String_data entity. It DOES NOT do anything
	// with the reference count - typically this will be 1
	String(String_data *s)				: sdp(s){}
public:
	//static const int debug;	// used for printfs
	String();
	String(const char *);
	String(char);
	String(int);
	String(unsigned);
	String(long);
	String(unsigned long);
	String(const String& s);
	String(const char *cp, size_t nbytes);	// create a substring
	~String();
	void operator	++();
	String & operator	=(const String & s);
	String & operator	=(const char *);
	String & operator	+=(char);
	String & operator	+=(const String&);
	String & operator	+=(const char *);
	friend String operator	+(const String&, const String&);
	friend String operator	+(const String&, const char *);
	friend String operator	+(const char *, const String&);

	friend int operator	==(const String&, const String&);
	friend int operator	==(const String&, const char *);
	friend int operator	==(const String&, char);
	friend int operator	==(char c, const String &csr){return csr==c;}
	friend int operator	!=(const String& csr, char c){ return !(csr == c);}
	friend int operator	!=(char c, const String &csr){return csr!=c;}
	friend int operator	>(const String&, const String&);
	friend int operator	>(const String&, const char *);
	friend int operator	<(const String&, const String&);
	friend int operator	<(const String&, const char *);

	char operator	[](size_t index1)const;	// character in String 1..n
	operator	int()const;
	operator	long()const{return ulongval();}
	const char * cstring()const{return sdp->cstring();}
	void check()const;
	String substr(size_t startpos, size_t length)const;	// characters count 1..n
	String tailstr(size_t startpos)const;		// as above
	String right(size_t n)const;			// The n rightmost chars
	String left(size_t n)const;			// The n leftmost
	void set_char(size_t index1, char cval);
	void set_substr(size_t startpos, const char * set_from, char filler = ' ');
	void set_substr(size_t spos, const String &set_from, char filler = ' '){
		set_substr(spos, set_from.sdp->data, filler);
	}
	void pad_to_length(size_t length, char filler = ' ');
	size_t nchars()const			{return strlen(sdp->data);}
	int is_null()const			{return sdp->data[0] == 0;}
	void cvt_toupper();
	void cvt_tolower();
	int nrefs()const			{return sdp->refcnt;}
	// and now, the equivalents of the character searching functions
	size_t strchr(char c)const;	// return index1 of first 'c' or 0
	size_t strrchr(char c)const;	// return index1 of last 'c' or 0
	size_t strstr(const String&)const; // index1 of 1st. occurrence of substr
	size_t strpbrk(const String &)const;	// index of 1st occ. of any char in arg

	// If the String holds digits (your responsibility), retrieve
	// the value of the brute, using strtoul etc
	unsigned long ulongval()const;
	unsigned long ulongval(int base)const;
	double doubleval()const;

	// Set the contents of a String as if by using sprintf.
	// WARNING!!!!! The maximum length of this is 512 bytes, and because
	// it really does use sprintf IT CANNOT BE CHECKED!!
	void sprintf(const char *, ...);

	static const char *NULLCHARSTRING;
	static String NULLSTRING;

	unsigned hashval()const			{ return hashval(sdp->data); }
        static unsigned hashval(const char *);

};

// Case-insensitive comparisons; return true for equality
int caseicmp(const char *, const char *);
int caseicmp(const char *, const String &);
int caseicmp(const String &, const String &);
int caseicmp(const String &, const char *);

inline int operator	!=(const String& cs1, const String& cs2){
	return !(cs1 == cs2);
}
inline int operator	!=(const String& cs1, const char * cs2){
	return !(cs1 == cs2);
}
inline int operator     ==(const String &csr, char c){
	return ((csr[1] == c) && (csr.nchars() == 1));
}
inline int operator	>=(const String& cs1, const String& cs2){
	return !(cs1 < cs2);
}
inline int operator	>=(const String& cs1, const char * cs2){
	return !(cs1 < cs2);
}
inline int operator	<=(const String& cs1, const String&cs2){
	return !(cs1 > cs2);
}
inline int operator	<=(const String& cs1, const char *cs2){
	return !(cs1 > cs2);
}

// Versions which take a char * as their LH arg
inline int operator	==(const char *cs1, const String &cs2){
	return cs2 == cs1;
}
inline int operator	!=(const char * cs1, const String& cs2){
	return !(cs2 == cs1);
}
inline int operator	>(const char *cs1, const String& cs2){
	return (cs2 < cs1);
}
inline int operator	>=(const char *cs1, const String& cs2){
	return (cs2 <= cs1);
}
inline int operator	<(const char *cs1, const String &cs2){
	return (cs2 > cs1);
}
inline int operator	<=(const char *cs1, const String&cs2){
	return (cs2 >= cs1);
}

String operator	+(const String &, char);
String operator	+(char, const String &);

// Return base 1 index position of first occurrence of 'c' in the String,
// or zero if there isn't one
inline size_t String::strchr(char c)const{
	char *cp = ::strchr(sdp->data, c);
	if(cp)
		return(unsigned int)(cp - sdp->data+1);
	return 0;
}

inline size_t String::strrchr(char c)const{
	char *cp = ::strrchr(sdp->data, c);
	if(cp)
		return(size_t)(cp - sdp->data+1);
	return 0;
}

// Return base 1 index position of first occurrence of contained subString
inline size_t String::strstr(const String &search)const{
	char *cp = ::strstr(sdp->data, search.cstring());
	if(cp)
		return(size_t)(cp - sdp->data+1);
	return 0;
}

// Return base 1 index position of first occurrence of any char from the arg
inline size_t String::strpbrk(const String &search)const{
	char *cp = ::strpbrk(sdp->data, search.cstring());
	if(cp)
		return (size_t)(cp-sdp->data+1);
	return 0;
}

class array_of_String : private sortable_list_of<String>{
	// void *implementation;	// trick - whatever is used, it's opaque
public:
	sortable_list_of<String>::element_count;
	sortable_list_of<String>::operator +=;
	sortable_list_of<String>::delete_this_element;
	sortable_list_of<String>::last_element;
	sortable_list_of<String>::delete_last_element;
	sortable_list_of<String>::insert_before_element;
	sortable_list_of<String>::sort;
	sortable_list_of<String>::clear;

	// looking at the code that Borland produces, we want this out-of-line
	array_of_String();
	//array_of_String(const array_of_String&);
	//virtual ~array_of_String();

	String &operator [](int index0);
	const String &operator[](int index0)const;

	// Remove & return first element in array
	String extract();

	// void operator =(const array_of_String &rhs);

	// void empty();	// delete all elements
	// Take a String and split it into fields, adding them to the
	// array. Return the number of fields. If you call strictgetfields, then
	// each delimiter counts and leading delimiters count (i.e, null fields
	// are expected). Ordinary getfields() ignores null fields. Strictgetfields
	// has no default, getfields defaults to whitespace separators.
	// NOTE: strictgetfields does NOT consider the sequence <delim><NULL> to be a field,
	// whereas a leading <delim> does imply an empty field. e.g strictgetfields(":a:", ":")
	// returns two fields: "" and "a". If you want a trailing empty field, use <delim><delim>
	int getfields(const String&csr, const char *delims);
	int getfields(const String &csr);
        int strictgetfields(const String &csr, const char *delims);

	// Search for a string - if found, return its index else -1
	enum findcase{honourcase, ignorecase};
	int findat(const String &csr, findcase f = honourcase);
	int findat(const char * csr, findcase f = honourcase);
};

// copy cp2 into cp - don't ovewrite the end or copy the null
// used when setting subStrings..
void copystr(char *cp, const char *cp2);


#endif
